--------------------------------------------------------------------
--            SymCACP Module 1         Golly bits
-- Symmetrical CA Control Panel   symCACP
--------------------------------------------------------------------
--  P. Rendell   12/10/2018
-- 16/01/2018  radical change to align seed definition in diagonal with orthogonal -- change vertically across boundary in both
-- 17/03/2022  change random to be 50/50 +-1
--------------------------------------------------------------------

local m ={}		-- class main
local rfw = {}		-- class RuleFileWriter
local g = golly()
local histNo = 0	-- id for history
local GlobalSwitch_Castle = false


--------------------------------------------------------------------
--------------------------------------------------------------------
--------------------------------------------------------------------------------
local function setRule(rule)
   local newRuleTxt
   function gsetRule()
      g.setrule(newRuleTxt)
   end
   local ruleBs
   local ruleHex
   if (rule:find("H")) then
      ruleBS = m.convHex2BS(rule)
      ruleHex = rule
   else
      ruleBS = rule
      ruleHex = m.convBS2Hex(rule)
   end
   local ruleHexNum = tonumber(ruleHex:upper():gsub('H',''),16)
   local threeState = (ruleHexNum%2 == 1)   
   if threeState then
      g.setalgo("RuleLoader")
      newRuleTxt = 'HEX'..string.format('%03x',ruleHexNum):upper()..':T'..m.width..','..m.hight
   else
      g.setalgo("HashLife")
      newRuleTxt = ruleBS..':T'..m.width..','..m.hight
   end
   
   local ok = true
   local status, err = pcall(gsetRule)
   if err then
      g.continue("")  -- don't show error when script finishes
      ok = false
   end
   if (ok) then   
      m.rule = rule
      m.ruleBS = ruleBS
      m.ruleHex = ruleHex
      m.threeState = threeState
   else
      g.warn("Rule "..rule.." not found Golly's Rules folder may be incorrect in preferences", false)
      local same = "no3sChg"; if (m.threeState ~= threeState) then same = "3sChg" end
      if (m.threeState ~= threeState) then
         if m.threeState then
            g.setalgo("RuleLoader")
         else
            g.setalgo("HashLife")
         end      
      end      
   end   
   return(ok)
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
-- init Function

function m.init(logFile, report, seed, geo)
   m.Switch_InvertIsland = false
   m.report = report
   m.logFile = logFile
   m.built = false
   m.loadRule("SymCACP ", tonumber(g.getwidth()), tonumber(g.getheight()), seed, geo, g.getrule():gsub("HEX","H"):gsub(":.*",""))
   m.gen = tonumber(g.getgen())
   g.update()
   m.logFile:write("buildUni ini rule "..m.rule.."\n")
end

--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function putCs(pat)
   i = 1
   while i < #pat do
      x = pat[i];i = i+1
      y = pat[i];i = i+1
      if m.threeState then
         s = pat[i]; i = i+1
         g.show("putting "..x..","..y..":"..s)
         g.setcell(x, y, s)
      else
         g.show("putting "..x..","..y)
         g.setcell(x, y, 1)
      end
   end
end
--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
--==================================================================
function m.GobalSwitch(switch, value)
   if(switch == "CASTLE") then
      GlobalSwitch_Castle = value
   end
end


--==================================================================
--     Utility Routines
--------------------------------------------------------------------

function cmpArrays(a,b)
   if #a ~= #b then
      return (false)
   end
   for i = 1, #a do
      if a[1] ~= b[i] then
         return (false)
      end
   end
   return (true)
end
--------------------------------------------------------------------------------
local function rleExpand(txt)
   local newTxt = ""
   local mult = 0
   local code
   for c in txt:gmatch"." do
       if c:find("[0-9]") then
         mult = mult*10+tonumber(c)
       else
          if( code) then
             newTxt = newTxt..(code:rep(mult-1))
          end
          code = c
          mult = 0
          newTxt = newTxt..c
       end
   end
   if( code) then
       newTxt = newTxt..(code:rep(mult-1))
   end
   return(newTxt)
end
--==================================================================
--     Golly Routines
--------------------------------------------------------------------

function m.doRun()
    while true do
        g.run(1)
        g.update()
        local event = g.getevent()
        if event:find("^key") or event:find("^oclick") then
            break
        end
    end
end
--------------------------------------------------------------------

function m.doStep()
   g.run(1)
   g.update()
end
--------------------------------------------------------------------

function m.doStepN(gens)
   g.run(gens)
   g.update()
end
--==================================================================
--     Universe State for Undi-Redo
--------------------------------------------------------------------
function m.GetUniverse()
   local uni = {}
   uni.isSame = function (other)
      return ( (uni.pop == other.pop) and  ( cmp(uni.universe, other.universe) == 0) )
   end -------------------------------------------------------------      
   uni.width = tonumber(g.getwidth())
   uni.hight = tonumber(g.getheight())
   g.update()
   uni.rule = g.getrule()     -- e.g B3/S01234678:T50,50
   uni.gen = tonumber(g.getgen())
   local pop = tonumber(g.getpop())
   uni.pop = math.min(pop, uni.width*uni.hight - pop)
   uni.threeState = uni.rule:find("H")
   if uni.width*uni.hight > 0 then
      uni.universe = g.getcells({-(uni.width//2), -(uni.hight//2), uni.width, uni.hight})
      m.logFile:flush()
   else 
      uni.universe = {}
   end
   return(uni)
end
--------------------------------------------------------------------

function m.SetUniverse(uni)
   if uni.threeState then
      g.setalgo("RuleLoader")
   else
      g.setalgo("HashLife")
   end
   g.setrule(uni.rule)    
   if uni.threeState then
      g.setcolors({1,48,48,48, 2,255,255,255})
   end
   m.logFile:flush()
   m.clearUni()
   g.putcells(uni.universe)
   g.setgen(tostring(uni.gen))
   g.update()
end
--------------------------------------------------------------------

--==================================================================
--==================================================================
--     CA Rule Routines
--------------------------------------------------------------------
function m.deriveSrule(bRule)    -- 'B12345678' -> 'S8'
   local sRule  = "012345678"
   local opChar = "876543210"
   for i = 2, string.len(bRule) do
      local c = bRule:sub(i,i)
      sRule = sRule:gsub(opChar:sub(tonumber(c)+1,tonumber(c)+1), "")
   end
   return('S'..sRule)
end
--------------------------------------------------------------------
function m.expandRuleString(bRule)  -- Bnnn -> Bnnn/Smmm
   local res = bRule
   if bRule:match('B[0-9]*') then
      res = bRule..'/'..m.deriveSrule(bRule)
   end
   return(res)
end
--------------------------------------------------------------------
function m.convBS2Hex(bRule)   -- 'Bnnn' -> Hnn
   local pow2 = {1,2,4,8,16,32,64,128,256}
   local v = 0
   local done = {}
   for i = 2, string.len(bRule:gsub('/.*','')) do
      local c = tonumber(bRule:sub(i,i))
      if not done[c] then
         if (c < 9) then
            v = v + pow2[c+1]
            done[c] = true
         else
            v = 513 -- invalid result
         end
      else
         v = 513 -- invalid result
      end
   end
   if bRule:find('S') then
      local newRule = m.convHex2BS('H'..string.format('%03x',v):upper())
      if bRule ~= newRule then
         m.report.collect('Rule c '..bRule..' not symmetric, using '..newRule..' instead', true)
      end
   end
   return('H'..string.format('%03x',v):upper())
end
--------------------------------------------------------------------
function m.convHex2BS(hexRuleP)    -- convert Hex rule to B/S string
   local rule=""
   local hexNum=tonumber(hexRuleP:upper():gsub('H',''),16)
   if not hexNum then
      hexNum = 0
   end
   if hexNum <512 then
      local testNum = hexNum
      local bit = 0
      local resListB = 'B'
      while testNum > 0 do
         if testNum%2 == 1 then
            resListB = resListB..(tostring(bit))
         end
         testNum = testNum//2
         bit = bit + 1
      end
      rule = m.expandRuleString(resListB)
   end
   return(rule)
end
--------------------------------------------------------------------
function m.hexFormat(rule)    -- puts a rule in correct HEX format
   if not rule:find("H") then
       return m.convBS2Hex(rule)
   end
   local r = rule
   if (#r < 2) then r = "H000" end
   if rule:sub(2):find('[^0-9A-F]') then
      r = rule:sub(2):gsub('[^0-9A-F]','')
      m.report.collect('Rule h '..rule..' has invalid characters using '..r..' instead', true)
   end
   return 'H'..string.format('%03x',tonumber(r:upper():gsub('H',''),16)):upper()
end
--------------------------------------------------------------------
function m.comp(rule)    -- convert rule to its compliment return in the same format
   local hexRule = rule
   if not rule:find("H") then
      hexRule = m.convBS2Hex(rule)
   end
   local compRule = 'H'..string.format('%03x',511 -tonumber(hexRule:upper():gsub('H',''),16)):upper()
   if not rule:find("H") then
      compRule = m.convHex2BS(rule)
   end
   return compRule
end

--==================================================================

--     CA universe Routines
--------------------------------------------------------------------
function makeDiag()
   local newPat = {}
   local ind = 1
   for i = -(m.width//2), (m.width-1)//2 do
      for j = 0, (m.hight-1)//2 do
         newPat[ind] = i
         newPat[ind+1] = (j-i + m.hight//2-(m.width)//2)%m.hight -(m.hight//2)
         ind = ind+2
         if m.threeState then
            newPat[ind] = 1
            ind = ind +1
         end
      end
      if m.threeState then
         for j = -(m.hight//2), -1 do
            newPat[ind] = i
            newPat[ind+1] =  (j-i + m.hight//2-(m.width)//2)%m.hight - (m.hight//2)
            newPat[ind+2] = 2
            ind = ind + 3
         end
      end
   end
   if m.threeState and ((ind-1)%2==0) then
      newPat[ind] = 0
   end
   g.putcells (newPat)
end
--------------------------------------------------------------------
function makeChecker()
   local newPat = {}
   local ind = 1
   for i = -(m.width//2), -1 do
      for j = 0, (m.hight-1)//2 do
         newPat[ind] = i
         newPat[ind+1] = j
         ind = ind+2
         if m.threeState then
            newPat[ind] = 1
            ind = ind +1
         end
      end
      if m.threeState then
         for j = -(m.hight//2), -1 do
            newPat[ind] = i
            newPat[ind+1] = j
            newPat[ind+2] = 2
            ind = ind + 3
         end
      end
   end
   for i = 0, (m.width-1)//2 do
      for j = -(m.hight//2), -1 do
         newPat[ind] = i
         newPat[ind+1] = j
         ind = ind+2
         if m.threeState then
            newPat[ind] = 1
            ind = ind +1
         end
      end
      if m.threeState then
         for j = 0, (m.hight-1)//2 do
            newPat[ind] = i
            newPat[ind+1] = j
            newPat[ind+2] = 2
            ind = ind + 3
         end
      end
   end
   if m.threeState and ((ind-1)%2==0) then
      newPat[ind] = 0
   end
   g.putcells (newPat)
end
--------------------------------------------------------------------
function makeOrthog()
   local newPat = {}
   local ind = 1
   for i = -(m.width//2), (m.width-1)//2 do
      for j = 0, (m.hight-1)//2 do
         newPat[ind] = i
         newPat[ind+1] = j
         ind = ind+2
         if m.threeState then
            newPat[ind] = 1
            ind = ind +1
         end
      end
      if m.threeState then
         for j = -(m.hight//2), -1 do
            newPat[ind] = i
            newPat[ind+1] = j
            newPat[ind+2] = 2
            ind = ind + 3
         end
      end
   end
   if m.threeState and ((ind-1)%2==0) then
      newPat[ind] = 0
   end
   g.putcells (newPat)
   if(GlobalSwitch_Castle or LocaSwitch_Castle) then
      m.makeOrthogCastle()
   end
end
--------------------------------------------------------------------
function m.makeOrthogCastle()
   local opposite={1,0}
   if m.threeState then
      opposite={0,2,1}
   end

   local y0=-1
   local y1 = (m.hight-1)//2
   for i = -(m.width//2), (m.width-1)//2,2 do
      g.setcell(i,y0,opposite[1+g.getcell(i,y0)]) 
      g.setcell(i,y1,opposite[1+g.getcell(i,y1)]) 
   end
end

--------------------------------------------------------------------
function makeRandomPattern(pWidth, pHight, filter, filterPar, percent)
   local opposite={1,0}
   if m.threeState then
      opposite={0,2,1}
      local newPat = {}
      local ind = 1
      for i = -(m.width//2), (m.width-1)//2 do
         for j = -(m.hight//2), (m.hight-1)//2 do
            newPat[ind] = i
            newPat[ind+1] = j
            newPat[ind+2] = 2
            ind = ind + 3
         end
      end
      if ((ind-1)%2==0) then
         newPat[ind] = 0
      end      
      g.putcells (newPat)
   end
   local newPat = {}
   local ind = 1
   local stateAcnt = 0
   local stateBcnt = 0
   for i = -(pWidth//2), (pWidth-1)//2 do
      for j = -(pHight//2), (pHight-1)//2 do
         local pt = filter(i, j, filterPar)
         if (m.Switch_InvertIsland) then pt.ok = not pt.ok end
         if (pt.ok) then
            if  (math.random() < percent/100) then
               newPat[ind] = pt.x
               newPat[ind+1] = pt.y
               ind = ind+2
               if m.threeState then
                  newPat[ind] = 1
                  ind = ind + 1
               end
               stateAcnt = stateAcnt + 1
            else
               stateBcnt = stateBcnt + 1
            end
         end
      end
   end
   if m.threeState and ((ind-1)%2==0) then
      newPat[ind] = 0
   end
   g.putcells(newPat)
   local i, j, temp
   local stateA = 1
   local stateB = opposite[stateA+1]
   if (stateAcnt/(stateAcnt+stateBcnt) > percent/100) then
      stateB = 1
      stateA = opposite[stateB+1]
      temp = stateAcnt
      stateAcnt = stateBcnt
      stateBcnt = temp
      percent = 100 - percent
   end
   local a0 = stateAcnt
   local b0 = stateBcnt
   while(stateAcnt/(stateAcnt+stateBcnt) < percent/100)do
      i = math.floor(math.random()*(pWidth-1) - (pWidth//2)+0.5)
      j = math.floor(math.random()*(pHight-1) - (pHight//2)+0.5)
      local pt = filter(i, j, filterPar)
      if (m.Switch_InvertIsland) then pt.ok = not pt.ok end
      if (pt.ok and (g.getcell(pt.x, pt.y) == stateB) ) then
         g.setcell(pt.x, pt.y, stateA)
         stateAcnt = stateAcnt + 1
         stateBcnt = stateBcnt - 1
      end
   end
   g.update()
end

--------------------------------------------------------------------
function islandFilterNil(x,y,radius)
   return({ok = true, x=x, y=y})
end

--------------------------------------------------------------------
function islandFilterDiamond(x,y,radius)
   local root2 = math.sqrt(2)
   local xv = math.floor((x+y)/root2 + 0.5)
   local yv = math.floor((y-x)/root2 + 0.5)
   return({ok = (xv >= -(m.width//2)) and (xv <= (m.width-1)//2) and (yv >= -(m.hight//2)) and (yv <= (m.hight-1)//2), x = xv , y = yv})
end

--------------------------------------------------------------------
function islandFilterRound(x,y,radius)
   if ( (x^2 + y^2) > (radius//2)^2 ) then
      return({ok = false, x=x, y=y})
   else
      return({ok = true, x=x, y=y})
   end
end

--------------------------------------------------------------------
function FilterDiag(x,y,radius)
--   if ( ((m.hight/2 - y) >= x) and ((m.hight/2 - y) < (m.width//2 + x))) then
--   if ((y >= -m.width/2 - x) and (y < - x)) or (y > m.width/2 + x ) then
--   if ((y >= - m.width/2 - x) and (y < - x)  ) or ((y <= x ) and (x > 0)) then
   if  ((y >= - m.width/2 - x) and (y <  - x)  ) or(( 1 +y - m.hight/2 > - x) and (x > 0) ) then
      return({ok = true, x=x, y=y})
   else
      return({ok = false, x=x, y=y})
   end
end

--------------------------------------------------------------------
function makeRandom()
   makeRandomPattern(m.width, m.hight, islandFilterNil, nil, tonumber(m.geo:match("%%([0-9]+)")))
end

--------------------------------------------------------------------   
function makeRandomDiag()
   makeRandomPattern(m.width,m.hight, FilterDiag, nil, tonumber(m.geo:match("%%([0-9]+)")))
end

--------------------------------------------------------------------   
function makeRandomIsland()
   local IsleSize = tonumber(m.geo:match("[0-9]+"))
   local percent = tonumber(m.geo:match("%%([0-9]+)"))
   local solid = m.geo:sub(-1) == "S"
   if (not IsleSize) then IsleSize = 0 end
   if ( (IsleSize < math.max(m.width,m.hight)) or (IsleSize < 2) ) then
      local opposite={2,1}
      g.show("makeRandomIsland "..m.geo)
      if (m.geo:sub(2,2) == "R") then
         if (m.Switch_InvertIsland) then
            makeRandomPattern(m.width, m.hight, islandFilterRound, IsleSize, percent)
         else
            makeRandomPattern(IsleSize, IsleSize, islandFilterRound, IsleSize, percent)
         end
      elseif (m.geo:sub(2,2) == "D") then
         makeRandomPattern(IsleSize, IsleSize, islandFilterDiamond, nil, percent)
      else
         makeRandomPattern(IsleSize, IsleSize, islandFilterNil, nil, percent)
      end
  else
      g.show("Island Size invalid")
   end
end

--------------------------------------------------------------------   
function addDiagSeed3()
   local x = -(m.width//2)
   local xB = m.width//2 - 1
   local y = 0
   local yN = -1
   local opposite={1,0}
   if m.threeState then
      opposite={0,2,1}
   end

   function incX(x)
      local nx = x+1
      if (nx > m.width//2 - 1) then
            nx = -(m.width//2)
      end
      return(nx)
   end
   
   function decY(y)
      local ny = y-1
      if (ny < -(m.hight//2)) then
         ny = m.hight//2 -1
      end
      return(ny)
   end
   
   digAction = {
      ['0'] = function() end,
      ['1'] = function() 
               g.setcell(x,y,opposite[1+g.getcell(x,y)]) 
            end,
      ['2'] = function()
               g.setcell(xB,yN,opposite[1+g.getcell(xB,yN)])
            end,
      ['3'] = function() 
               g.setcell(x,y,opposite[1+g.getcell(x,y)])
               g.setcell(xB,yN,opposite[1+g.getcell(xB,yN)])
            end
   }
   

   for i = 1, #m.seed do
      --g.show("Pt. "..x..","..y.." Op "..xB..","..yN)
      local dig = m.seed:sub(i,i)
      if dig:match("[0123]") then
         digAction[dig]()
         if (i%2 == 1) then
            xB = x
            x = incX(x)
         else
            y = yN
            yN = decY(y)
         end
     -- g.show("Pt "..x..","..y.." Op "..xB..","..yN)
      else
         m.logFile:write("*** Invalid digit "..dig.." in seed "..(m.seed).." ***\n")
         m.logFile:flush()
      end
   end
end
--------------------------------------------------------------------   

function addDiagSeed()
   local opposite={1,0}
   if m.threeState then
      opposite={0,2,1}
   end
   
   local x = -(m.width//2)
   local y = -1
   function incX(x)
      x = x+1
      if (x > m.width//2 - 1) then
            x = -(m.width//2)
      end
      return(x)
   end
   function incY(y)
      y = y+1
      if (y > m.hight//2 - 1) then
            y = -(m.hight//2)
      end
      return(y)
   end
   digAction = {
      ['0'] = function() end,
      ['1'] = function() 
               g.setcell(x,y,opposite[1+g.getcell(x,y)]) 
            end,
      ['2'] = function()
               g.setcell(incX(x),incY(y),opposite[1+g.getcell(x,incY(y))])
            end,
      ['3'] = function() 
               g.setcell(x,y,opposite[1+g.getcell(x,y)])
               g.setcell(incX(x),incY(y),opposite[1+g.getcell(x,incY(y))])
            end
   }
   
   for i = 1, #m.seed do
      local dig = m.seed:sub(i,i)
      if dig:match("[0123]") then
         digAction[dig]()
         x = incX(x)
         y = y - 1
         if (y < -(m.hight//2)) then
            y = (m.hight-1)//2 
         end
      else
         m.logFile:write("*** Invalid digit "..dig.." in seed "..(m.seed).." ***\n")
         m.logFile:flush()
      end
   end
end
--------------------------------------------------------------------   
   
function addOrthogSeed(offSet)
   local opposite={1,0}
   if m.threeState then
      opposite={0,2,1}
   end
   
   local x = -(m.width//2)+offSet
   if (x > m.width//2 - 1) or (x < -(m.width//2)) then
      x = -(m.width//2)
   end
   local y = -1
   function incY(yp)
      local y = yp+1
      if (y > m.hight//2 - 1) then
            y = -(m.hight//2)
      end
      return(y)
   end
   digAction = {
      ['0'] = function() end,
      ['1'] = function() 
               g.setcell(x, y, opposite[1+g.getcell(x,y)]) 
            end,
      ['2'] = function()
               g.setcell(x, incY(y), opposite[1+g.getcell(x,incY(y))])
            end,
      ['3'] = function() 
               g.setcell(x, y, opposite[1+g.getcell(x,y)])
               g.setcell(x, incY(y), opposite[1+g.getcell(x,incY(y))])
            end
   }
   
   for i = 1, #m.seed do
      local dig = m.seed:sub(i,i)
      if dig:match("[0123]") then
         digAction[dig]()
         x = x+1
         if (x > m.width//2 - 1) then
            x = -(m.width//2)
         end
      else
         m.logFile:write("*** Invalid digit "..dig.." in seed "..(m.seed).." ***\n")
         m.logFile:flush()     
      end
   end
end
--------------------------------------------------------------------------------

function m.doRotateRL(dir)
   local colTemp = {}
   local x0 = (m.width-1)//2
   local x1 = -(m.width//2)
   local inc = -1
   if (dir ~= "r") then
      x0 = x1
      x1 = (m.width-1)//2
      inc = 1
   end
   for j = -(m.hight//2), (m.hight-1)//2 do
      colTemp[j] = g.getcell(x0, j)
   end
   i = x0
   for n = 1, m.width-1 do
      i = i + inc
      for j = -(m.hight//2), (m.hight-1)//2 do
         g.setcell(i-inc, j, g.getcell(i, j))
      end
   end
   for j = -(m.hight//2), (m.hight-1)//2 do
      g.setcell(x1, j, colTemp[j])
   end
end
--------------------------------------------------------------------------------

function m.doRotateUD(dir)
   local rowTemp = {}
   local y0 = (m.hight-1)//2
   local y1 = -(m.hight//2)
   local inc = -1
   if (dir == "u") then
      y0 = y1
      y1 = (m.hight-1)//2
      inc = 1
   end
   for i = -(m.width//2), (m.width-1)//2 do
      rowTemp[i] = g.getcell(i, y0)
   end
   j = y0
   for n = 1, m.hight-1 do
      j = j + inc
      for i = -(m.width//2), (m.width-1)//2 do
         g.setcell(i, j-inc, g.getcell(i, j))
      end
   end
   for i = -(m.width//2), (m.width-1)//2 do
      g.setcell(i, y1, rowTemp[i])
   end
end
--------------------------------------------------------------------------------

function m.doRotateUp()
   local rowTemp = {}
   local y = -(m.hight//2)
   for i = -(m.width//2), (m.width-1)//2 do
      rowTemp[i] = g.getcell(i, y)
   end
   for i = -(m.width//2), (m.width-1)//2 do
      for j = -(m.hight//2), (m.hight-3)//2 do
         g.show("rr i,j "..i..','..j)
         g.setcell(i, j, g.getcell(i, j+1))
      end
   end
   y = (m.hight-1)//2
   for i = -(m.width//2), (m.width-1)//2 do
      g.setcell(i, y, rowTemp[i])
   end
end
--------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------
-- symmetrical along orthoganal boundry opposite states
local function testMirrorB(seed, mirrorChars)
   symmetrical = true
   if (#seed > 1) then
      for i = 1,#seed do
         symmetrical = mirrorChars[seed:sub(i,i)]
         if( not symmetrical) then
            break
         end
      end
   end
   return(symmetrical)
end
--------------------------------------------------------------------------------------------

local function testMirror(seed, seedI1, mirrorPairs, ty)
   symmetrical = (seedI1 > 0)
   for i = 1,seedI1 do	
      pair = string.sub(seed,i,i)..string.sub(seed,#seed+1-i,#seed+1-i)
      symmetrical = mirrorPairs[pair]
      if(not symmetrical) then
         break
      end
   end
   return(symmetrical)
end
--------------------------------------------------------------------------------------------
-- diagonal boundry  : rotation even length , palandrome odd length
function m.seedSym(seedParm)
   SymetricRotate = 
         {["00"]=true,  ["01"]=false, ["02"]=false, ["03"]=false,
          ["10"]=false, ["11"]=false, ["12"]=true,  ["13"]=false,
          ["20"]=false, ["21"]=true,  ["22"]=false, ["23"]=false,
          ["30"]=false, ["31"]=false, ["32"]=false, ["33"]=true   }
   SymetricMirrorPerP = 
         {["00"]=true,  ["01"]=false, ["02"]=false, ["03"]=false,
          ["10"]=false, ["11"]=true,  ["12"]=false, ["13"]=false,
          ["20"]=false, ["21"]=false, ["22"]=true,  ["23"]=false,
          ["30"]=false, ["31"]=false, ["32"]=false, ["33"]=true   }
   SymetricMirrorBoundry = 
         {["0"]=true, ["1"]=false, ["2"]=false, ["3"]=true}
   SymetricCenter = 
         {["0"]="prb", ["1"]="pb", ["2"]="pb", ["3"]="prb"}
   
   local seedP = seedParm:lower()
   if (string.find(seedP, "[oabc]")) then
      seedP = rleExpand(seedP):gsub('o','0'):gsub('a','1'):gsub('b','2'):gsub('c','3')
   end
   
   while (string.sub(seedP,1,1) == '0') do
      seedP = string.sub(seedP,2)
   end
   while (string.sub(seedP,#seedP) == '0') do
      seedP = string.sub(seedP,1,#seedP-1)
   end
   
   ret=false
   local centre
   if(#seedP >0 ) then
      centre = "prb-"
      if (#seedP%2 == 1) then
         centre = SymetricCenter[seedP:sub(#seedP//2+1,#seedP//2+1)]
      end
      if (centre:match("p")) then 
         ret = ret or testMirror(seedP, #seedP//2, SymetricMirrorPerP, "p")
      end
      if (centre:match("r")) then 
         ret = ret or testMirror(seedP, #seedP//2, SymetricRotate, "r")
      end
      if (centre:match("b")) then 
         ret = ret or testMirrorB(seedP, SymetricMirrorBoundry)
      end
   end
   return ret
end
--------------------------------------------------------------------------------------------

function m.clearUni()
   local ans = false
   if (not g.empty()) then
      g.select({0,0,1,1})
      g.clear(1)
      g.clear(0)
      g.select({})
      ans = true
   end
   return(ans)
end
--------------------------------------------------------------------

function m.loadRule(title, width, hight, seed, geo, rule)
   local ok = false
   m.width = tonumber(width)
   m.hight = tonumber(hight)
   if (m.hight == 0) then
      m.hight = m.width
   elseif (m.width == 0) then
      m.width = m.hight
   end
   if (m.width < 4) then
      m.width = 4
   end
   if (m.hight < 4) then
      m.hight = 4
   end
   if (setRule(rule)) then
      ok = true
      m.clearUni()
      g.new('SymCACP '..rule..' '..seed..' ')
      m.title = title
      m.seed = seed:lower()
      if (string.find(m.seed, "[oabc]")) then
         m.seed = rleExpand(m.seed):gsub('o','0'):gsub('a','1'):gsub('b','2'):gsub('c','3')
      end

      m.geo = geo
      if(geo == "OC") then
         m.geo = "O"
         LocaSwitch_Castle = true
      else
         LocaSwitch_Castle = false
      end
      m.logFile:write('loadRule '..' w '..m.width..' h '..m.hight..' s '..seed..' g '..geo..' r '..rule..' set '..g.getrule()..'\n')
      m.logFile:flush()
      if (m.threeState) then
         g.setcolors({1,48,48,48, 2,255,255,255})
      end
   end
   return(ok)
end
--------------------------------------------------------------------

local function conv2s3s(pat, swap)
   local map = {}
   local newPat = {}
   local targState = 1
   if (swap) then targState = 2 end
   if (#pat%2 == 1) then
      g.show("conv2s3s ->2 "..g.getpop()) --#t#
      for ind = 1, #pat, 3 do
         if (pat[ind+2] == targState) then
            newPat[#newPat+1] = pat[ind]
            newPat[#newPat+1] = pat[ind+1]
         end
      end
   else   
      for i = -(m.width//2), (m.width-1)//2 do
         map[i] = {}
      end
      local cnt = 0 --#t#
      for ind = 1, #pat, 2 do
         map[pat[ind]][pat[ind+1]] = true
         cnt = cnt + 1  --#t#
      end
      g.show("conv2s3s ->3 cnt "..cnt) --#t#
      local thisState
      local cnt2 = 0 --#t#
      for i = -(m.width//2), (m.width-1)//2 do
         for j = -(m.hight//2), (m.hight-1)//2 do
            thisState = targState
            if  (not map[i][j]) then thisState = 3 - targState; cnt2 = cnt2+1 end
            newPat[#newPat + 1] = i
            newPat[#newPat + 1] = j
            newPat[#newPat + 1] = thisState
         end
      end
      g.show("conv2s3s ->3 cnt "..cnt.." c2 "..cnt2) --#t#
      if (#newPat%2==0) then
         newPat[#newPat + 1] = 0
      end
   end
   return(newPat)
end

--------------------------------------------------------------------
function m.doSwapComp(title, width, hight, seed, geo, rule)
   local patt
   local gen = tonumber(g.getgen())
   local isPat = (not g.empty())
   if (isPat) then 
      patt = g.getcells({-(m.width//2), -(m.hight//2), m.width, m.hight})
   end
   local ruleHexNum = 511 -tonumber(m.ruleHex:upper():gsub('H',''),16)
   local threeState = (ruleHexNum%2 == 1)
   local ruleHex = 'H'..string.format('%03x',ruleHexNum):upper()
   local ruleBS = m.convHex2BS(ruleHex)
   local newRule = ruleHex
   if  ruleFMThex then
      newRule =  ruleBS
   end
   if (m.loadRule(title, m.width, m.hight, m.seed, m.geo, newRule)) then 
      if (isPat) then
         g.putcells(conv2s3s(patt, gen%2 == 1))
      end
   end
   g.setgen(gen)
   return (m.rule)
end

--------------------------------------------------------------------
function m.doCopy()
   local ans = true
   if (not g.empty()) then
      g.select({-(m.width//2), -(m.hight//2), m.width, m.hight})
      g.copy()
      g.select({})
   else
      ans = false
   end
   return(ans)
end
--------------------------------------------------------------------

function m.convert2ThreeState(patP, pWidth)
   local newPat = ""
   local pat = patP:gsub('[oO]','A'):gsub('b','B')
   for line in string.gmatch(pat, "([^$!]*)$?!?") do
      local leng = 0
      local cnt = 0
      for i = 1,#line do
         local c = line:sub(i,i)
         if (c:match("[0-9]")) then
            cnt = cnt * 10 + tonumber(c)
         elseif( c:match("[AB]")) then
            if (cnt == 0) then
               leng = leng + 1
            else
               leng  = leng + cnt
               cnt = 0
            end
         end
      end
      if (leng < pWidth) then
         local dif = pWidth - leng
         if (dif > 1) then
            line = line..tostring(dif)..'B'
         else
            line = line..'B'
         end
      end
      newPat = newPat..line..'$'
   end
   return(newPat..'!')
end
--------------------------------------------------------------------

function m.doPaste(swap)
   local pat = g.getclipstr()
   local x = tonumber(string.match(pat, '[xX] *= *([0-9\n]*)'))
   local y = tonumber(string.match(pat, '[yY] *= *([0-9\n]*)'))
   local ans = x and y and (x <= m.width) and (y <= m.hight)
   if (ans) then
      local ruleHexNum=tonumber(m.ruleHex:upper():gsub('H',''),16)
      local rTxt
      if m.threeState then
         rTxt = ', rule = HEX'..string.format('%03x',ruleHexNum):upper()..':T'..m.width..','..m.hight
      else
         g.setalgo("HashLife")
         rTxt = ', rule = '..m.ruleBS..':T'..m.width..','..m.hight
      end      
      pat = pat:match('(.*y[^,\n]*)')..rTxt..'\n'..pat:match('[^\n]*\n(.*)')
      local ts = "2State";if (m.threeState) then ts = "3State" end
      if (m.threeState) then
         pat = m.convert2ThreeState(pat, x)
         g.setclipstr(pat)
      else
         g.setclipstr(pat:gsub('[aA]','o'):gsub('B','b'))
      end
      g.paste(-(m.width//2), -(m.hight//2), "copy")
   else
      if ( x and y) then
         g.note("Pattern size "..x..','..y.." too big")
      else
         g.note("No Pattern to paste")
      end
   end
   return(ans)
end
--------------------------------------------------------------------------------

function m.doBuild(title, width, hight, seed, geo, rule)
   if (m.loadRule(title, width, hight, seed, geo, rule)) then
      g.show('doBuild '..' w '..width..' h '..hight..' s '..seed..' g '..geo..' r '..rule)
      -- geo has become complex  
      -- D...	Diagonal Band
      -- O...	Orthoganal Band
      -- R..	Random fill Universe
      -- IRnn%mm	Round Island size nn Random fill %mm
      -- ISnn%mm	Square Island size nn Random fill %mm
      -- IDnn%mm	Diamond Island size mm Random Fill %mm
      -- RANDOM_DIAG	Random Diagonal
      if m.geo:sub(1,1) == 'D' then
         makeDiag()
         if (m.seed:sub(1,1) == "-") then
            addDiagSeed()
         else
            addDiagSeed3()
         end
      elseif m.geo:sub(1,1) == 'O' then
         m.seed = m.seed:gsub('-','')
         makeOrthog()
         addOrthogSeed(0)
      elseif m.geo:sub(1,1) == 'C' then
         m.seed = m.seed:gsub('-','')
         makeChecker()
         addOrthogSeed(m.width//4-#m.seed//2)
      elseif m.geo:sub(1,1) == 'I' then
         g.show("builduni geo = "..geo.." m.geo ("..m.geo..")")
         m.seed = m.seed:gsub('-','')
         math.randomseed(m.seed)
         if(not m.geo:match("%%")) then  m.geo = m.geo.."%50" end
         makeRandomIsland()
      elseif m.geo:sub(1,1) == 'R' then
         m.seed = m.seed:gsub('-','')
         math.randomseed(m.seed)
         if(not m.geo:match("%%")) then  m.geo = m.geo.."%50" end
         if m.geo:sub(1,11):upper() == 'RANDOM_DIAG' then
            makeRandomDiag()
         else
            makeRandom()
         end
      end
      g.select({-(m.width//2),-(m.hight//2),m.width,m.hight})
      g.fitsel()
      g.setgen('0')
      g.select({})
      g.update()
      m.built = true
   else
      g.warn("loadrule failed")
   end
end
--------------------------------------------------------------------

function m.Fluid(pattern, stepX, offsetY)
   local opposite={1,0}
   if m.threeState then
      opposite={0,2,1}
   end
   local x = 0
   local start = true
   m.logFile:write('+++Fluid '..' pattern '..pattern..' #stepX '..#stepX..' = '..stepX[1]..' #offsetY '..#offsetY..' = '..offsetY[1]..'\n')
   while (start or (x < 0)) do
      for off = 1,#stepX do
         x = x + stepX[off]
         if (x > m.width//2 - 1) then
            x = x - m.width
            start = false
         end
         local s = "n"
         if (start)then  s = "Y" end
         if (start or (x < 0)) then
            local y = offsetY[off]
            local startY = true
            local sy = "n"
   m.logFile:write('+++Fluid '..' x '..x..' y '..y..' off '..off..'\n')
            if(startY) then sy = "Y" end
            while (startY or ( y < 0)) do
               for i = 1, #pattern do
	          local c = pattern:sub(i,i)
	          if (c == '1') then
                     g.setcell(x,y,opposite[1+g.getcell(x,y)]) 
--   m.logFile:write('+++Fluid '..' x '..x..' y '..y..'\n')
	          end
	          y = y + 1
	          if (y > m.hight//2 - 1) then
	             y = y - m.hight
	             startY = false
                  end
               end
            end
         end
      end
   end
   g.update()
   g.note("fluid")
end
--------------------------------------------------------------------
return m

--------------------------------------------------------------------
--                      	END OF FILE
--------------------------------------------------------------------
